#include <config.h>

#ifdef HAVE_ZLIB
#include <zlib.h>
#define TRANSFORM_UNUSED(x) x
#else
#define TRANSFORM_UNUSED(x) UNUSED(x)
#endif /* HAVE_ZLIB */

#include "transformation.h"

/*
 * Initialize
 */
static int
zlib_initialize(UNUSED(Transformation *xform), UNUSED(int enc))
{
	return 0;
}

/*
 * Shut down.
 */
static int
zlib_shutdown(UNUSED(Transformation *xform))
{
	return 0;
}

/*
 * Handle forks.
 */
static int
zlib_startNewTape(UNUSED(Transformation *xform), UNUSED(struct tapebuf *bin),
	UNUSED(unsigned long *destlen))
{
	return 0;
}

/*
 * Start slave process
 */
static int
zlib_startDiskIOProcess(UNUSED(Transformation *xform))
{
	return 0;
}

/*
 * End slave process
 */
static int
zlib_endDiskIOProcess(Transformation *xform)
{
	if (xform != NULL) {
		free(xform);
	}

	return 0;
}

struct req {
	ext2_loff_t dblk;
	int count;
};

/*
 * Compress a buffer.
 */
static int
zlib_compress(TRANSFORM_UNUSED(Transformation *xform), TRANSFORM_UNUSED(struct tapebuf *tpbin),
		TRANSFORM_UNUSED(unsigned long *destlen), TRANSFORM_UNUSED(const char *src), TRANSFORM_UNUSED(int srclen))
{
#ifdef HAVE_ZLIB
	int compresult;
	compresult = compress2((unsigned char*)tpbin->buf, destlen, (const unsigned char*)src, srclen, xform->state.zlib.complvl);
	return compresult == Z_OK ? 1 : 0;
#else
	return 1;
#endif /* HAVE_ZLIB */
}

/*
 * Decompress a buffer.
 */
static int
zlib_decompress(UNUSED(Transformation *xform), TRANSFORM_UNUSED(struct tapebuf *tpbin),
	TRANSFORM_UNUSED(unsigned long *destlen), TRANSFORM_UNUSED(const char *src), TRANSFORM_UNUSED(int srclen), TRANSFORM_UNUSED(char **reason))
{
#ifdef HAVE_ZLIB
	int cresult;
	cresult = uncompress((unsigned char*)tpbin->buf, destlen, (const unsigned char*)src, srclen);
	switch (cresult) {
		case Z_OK:
			*reason = "";
			break;
		case Z_MEM_ERROR:
			*reason = "not enough memory";
			break;
		case Z_BUF_ERROR:
			*reason = "buffer too small";
			break;
		case Z_DATA_ERROR:
			*reason = "data error";
			break;
		default:
			*reason = "unknown";
	}
	return (cresult == Z_OK) ? 1 : 0;
#else
	return 1;
#endif /* HAVE_ZLIB */
}


/*
 * Factory
 */
Transformation
*transformation_zlib_factory(int enc, TRANSFORM_UNUSED(int complvl))
{
	Transformation *t = (Transformation *) malloc(sizeof (Transformation));

	t->enc = enc;
#ifdef HAVE_ZLIB
	t->state.zlib.complvl = complvl;
#endif

	t->name = "zlib";
	t->mandatory = 0;
	t->initialize = &zlib_initialize;
	t->shutdown = &zlib_shutdown;
	t->startNewTape = &zlib_startNewTape;
	t->startDiskIOProcess = &zlib_startDiskIOProcess;
	t->endDiskIOProcess = &zlib_endDiskIOProcess;
	t->compress = &zlib_compress;
	t->decompress = &zlib_decompress;

	return t;
}
