// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System.Diagnostics.CodeAnalysis;
using System.Runtime.Versioning;

namespace System.Security.Cryptography
{
    /// <summary>
    ///   Provides a Cryptography Next Generation (CNG) implementation of Composite ML-DSA.
    /// </summary>
    /// <remarks>
    ///   <para>
    ///     Developers are encouraged to program against the <see cref="CompositeMLDsa" /> base class,
    ///     rather than any specific derived class.
    ///     The derived classes are intended for interop with the underlying system
    ///     cryptographic libraries.
    ///   </para>
    /// </remarks>
    [Experimental(Experimentals.PostQuantumCryptographyDiagId, UrlFormat = Experimentals.SharedUrlFormat)]
    public sealed partial class CompositeMLDsaCng : CompositeMLDsa
    {
        /// <summary>
        ///   Initializes a new instance of the <see cref="CompositeMLDsaCng"/> class by using the specified <see cref="CngKey"/>.
        /// </summary>
        /// <param name="key">
        ///   The key that will be used as input to the cryptographic operations performed by the current object.
        /// </param>
        /// <exception cref="ArgumentNullException">
        ///   <paramref name="key"/> is <see langword="null"/>.
        /// </exception>
        /// <exception cref="ArgumentException">
        ///   <paramref name="key"/> does not specify a Composite ML-DSA group.
        /// </exception>
        /// <exception cref="PlatformNotSupportedException">
        ///   Cryptography Next Generation (CNG) classes are not supported on this system.
        /// </exception>
        [SupportedOSPlatform("windows")]
        public CompositeMLDsaCng(CngKey key)
            : base(AlgorithmFromHandle(key))
        {
            throw new PlatformNotSupportedException();
        }

        private static CompositeMLDsaAlgorithm AlgorithmFromHandle(CngKey key) =>
            throw new PlatformNotSupportedException();

        /// <summary>
        ///   Gets a new <see cref="CngKey" /> representing the key used by the current instance.
        /// </summary>
        /// <exception cref="ObjectDisposedException">
        ///   This instance has been disposed.
        /// </exception>
        /// <remarks>
        ///   This <see cref="CngKey"/> object is not the same as the one passed to <see cref="CompositeMLDsaCng(CngKey)"/>,
        ///   if that constructor was used. However, it will point to the same CNG key.
        /// </remarks>
        public partial CngKey GetKey();
    }
}
